//
// Copyright (C) 2006 Henning Westerholt
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
//

/*
 * Changelog:
 * Version 18: Bugfix in xml generation, wrong case for the toPort attribute.
 *             Add a correct return value in main.
 * Version 19: Fix uplink port generation in network class, add a special
 *             case for the root switch. Now we have a correct port numbering
 *             and xml generation.
 * Version 20: Fix percent input, zero is valid again. Add help about the
 *             network size, there exist always a odd number of nodes.
 * Version 21: Add normal (no real-time) device generation. Small changes in
 *             information output.
 */

#include <iostream>
#include <stdlib.h>
#include <string.h>
#include <application.h>
#include <utility.h>
#include <QString>

using namespace std;

static const unsigned int version = 21;

void printHelp()
{
    cout << "Create random networks in the 'PG' format. " << "\n";
    cout << "Usage: ./createNetwork [FILE] [OPTIONS] [RANDOM SEED]" << "\n";
    cout << "\n";
    cout << "The [FILE] specify the output filename, this is a mandatory" << "\n";
    cout << "value." << "\n";
    cout << "\n";
    cout << "Valid [OPTIONS] are:" << "\n";
    cout << "Network size, the minimal size is 10, maximal is 100000." << "\n";
    cout << "Default value for the network size is 100. The size of the" << "\n";
    cout << "created network is always a odd value, for internal reasons." << "\n";
    cout << "\n";
    cout << "The four following parameters are percent values, the minimal" << "\n";
    cout << "value is therefore zero, the maximal one hundred." << "\n";
    cout << "\n";
    cout << "Sibling chance specify the network topology, use zero for a bus" << "\n";
    cout << "network, 100 for a star topology." << "\n";
    cout << "Default value for sibling chance is 50." << "\n";
    cout << "\n";
    cout << "Connection chance specify the chance that a node is connected" << "\n";
    cout << "to another node. 33 is the default value. A value of 100 percent" << "\n";
    cout << "don't work with random traffic mode at the moment, high chances" << "\n";
    cout << "leads also to multiple connections to the same receiver." << "\n";
    cout << "\n";
    cout << "Connection range specify how far the other connection partner" << "\n";
    cout << "are from a given node. 33 is a good start value, this is also" << "\n";
    cout << "the default." << "\n";
    cout << "\n";
    cout << "No real-time chance is the chance that a device is a normal (no" << "\n";
    cout << "real-time) device. 0-5 percent are meaningful values, 0 is the" << "\n";
    cout << "default. Network with no real-time devices are currently not" << "\n";
    cout << "supported by the schedule calculation application." << "\n";
    cout << "\n";
    cout << "Communication mode is the mode of the the generated communication." << "\n";
    cout << "The possible modes are 'random' (one to one) for a random network, " <<"\n";
    cout << "'master' for a master (one to many), and 'multimaster' (few to many) " << "\n";
    cout << "structure. Random is the default." << "\n";
    cout << "\n";
    cout << "[RANDOM SEED] specify the seed of the random generator. The same" << "\n";
    cout << "seed leads to the same networks (if the options are the same)." << "\n";
    cout << "Use a positive value from 0 to 1000000. Omit this value to use" << "\n";
    cout << "a seed generated from the system time and date." << "\n";
    cout << "\n";
    cout << "Examples:" << "\n";
    cout << "Create a medium network with mixed topology (default values):" << "\n";
    cout << "./createNetwork outputFile.xml" << "\n";
    cout << "\n";
    cout << "Create a big network with star topology, and few random connections:" << "\n";
    cout << "./createNetwork outputFile.xml 10000 100 15 33 0 random" << "\n";
    cout << "\n";
    cout << "Create a default network with a given random seed:" << "\n";
    cout << "./createNetwork outputFile.xml 34234" << "\n";
    cout << "\n";
    cout << "\n";
    cout << "createNetwork version " << version << "," << "\n";
    cout << "Copyright (C) 2006, Henning Westerholt," << "\n";
    cout << "E-Mail: 'hw at skalatan dot de'" << "\n";
    cout << "\n";
    cout << "createNetwork comes with ABSOLUTELY NO WARRANTY." << "\n";
    cout << "This is free software, and you are welcome to redistribute" << "\n";
    cout << "it under certain conditions." << "\n";
    cout << "\n";
    cout << "This program uses additional free software:" << "\n";
    cout << "- the STL-like templated tree class from Kasper Peeters" << "\n";
    cout << "- MersenneTwister class from Makoto Matsumoto and Takuji Nishimura" << "\n";
    cout << "\n";
    cout << "Please refer to the header of the source code for the copyright of " << "\n";
    cout << "certain files." << "\n";
}

void parseArguments(const int argc, const char * argv[], NetworkParameter& parameter, QString& fileName)
{
    // parse command line arguments - not elegant, but it works
    switch (argc) {
        case 2: case 3: {
            fileName = argv[1];
            // the user give us a random seed
            if (argc == 3)
                parameter.randomSeed = strtol(argv[2], static_cast<char **>(NULL), 10);
            else
                parameter.randomSeed = 0; // generate a random seed in the utility class

            // create a medium sized mixed topology,
            // if nothing is specified
            parameter.networkSize = 100;
            parameter.siblingChance = 0.5;
            parameter.connectionChance = 0.33;
            parameter.connectionRange = 0.33;
            parameter.normalDeviceChance = 0.00;
            parameter.trafficMode = Random;

            parameter.valid = true;
            break;
            }
        case 8: case 9: {
            fileName = argv[1];
            parameter.networkSize = strtol(argv[2], static_cast<char **>(NULL), 10);
            parameter.siblingChance = static_cast<double> (strtol(argv[3], static_cast<char **>(NULL), 10)) / 100;
            parameter.connectionChance = static_cast<double> (strtol(argv[4], static_cast<char **>(NULL), 10)) / 100;
            parameter.connectionRange = static_cast<double> (strtol(argv[5], static_cast<char **>(NULL), 10)) /100;
            parameter.normalDeviceChance = static_cast<double> (strtol(argv[6], static_cast<char  **>(NULL), 10)) /100;

            if (strcmp("master", argv[7]) == 0)
                parameter.trafficMode = Master;
            else if (strcmp("multimaster", argv[7]) == 0)
                parameter.trafficMode = MultiMaster;
            else if (strcmp("random", argv[7]) == 0)
                parameter.trafficMode = Random;
            else {
                parameter.valid = false;
                break;
            }

            //check argument ranges
            if ((parameter.networkSize >= 10 && parameter.networkSize <= 100000) &&
                Utility::isPercent(parameter.siblingChance) && 
                Utility::isPercent(parameter.connectionChance) &&
                Utility::isPercent(parameter.normalDeviceChance) &&
                Utility::isPercent(parameter.connectionRange)) {
                    parameter.valid = true;
                }
            else {
                parameter.valid = false;
                break;
            }
            // the user give us a random seed
            if (argc == 9) {
                parameter.randomSeed = strtol(argv[8], static_cast<char **>(NULL), 10);
            }
            break;
        }
        default: {
            parameter.valid = false;
            break;
        }
    }
}

void printArguments(const NetworkParameter& parameter, const QString& fileName)
{
    char * trafficMode = "";
    switch (parameter.trafficMode) {
        case Random: {
            trafficMode = "random";
            break;
        }
        case Master: {
            trafficMode = "master-slave";
            break;
        }
        case MultiMaster: {
            trafficMode = "multimaster";
            break;
        }
        default: // this should not happen
            break;
    }
    cout << "Create random network in file " << fileName.toStdString() << ", with size " << parameter.networkSize
        << ", sibling chance " << parameter.siblingChance << ", connection chance " << parameter.connectionChance
        << ", connection range " <<  parameter.connectionRange
        << ", normal device chance " << parameter.normalDeviceChance; 
    if (parameter.randomSeed == 0)
        cout << " and a auto generated random seed." << "\n";
    else
        cout << " and random seed " << parameter.randomSeed << ".\n";
    cout << "Use a " << trafficMode << " communication structure."  << "\n";
}

int main(const int argc, const char *argv[])
{
    auto_ptr<Application> myApp(new Application());

    QString fileName;
    NetworkParameter parameter;

    parseArguments(argc, argv, parameter, fileName);

    if (parameter.valid) {
        myApp->createNetwork(parameter, fileName);
        printArguments(parameter, fileName);
        exit(0);
    }
    else {
        printHelp();
        exit(1);
    }
}
