/*
 * Copyright (C) 2006 Henning Westerholt
 * Copyright (C) 2003 CTIE, Monash University
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#ifndef _MACRELAYUNITNPTDMA_H
#define _MACRELAYUNITNPTDMA_H


#include <vector>
#include "MACRelayUnitNP.h"

class EtherFrame;
class cXMLElement;

/**
 * holder class for schedule vector
 */
class IOHolder
{
  public:
    IOHolder() : inputPort(0), outputPort(0) {}
    IOHolder(const unsigned int inputPort, const unsigned int outputPort) : inputPort(inputPort), outputPort(outputPort) {}
    unsigned int getInputPort() {return inputPort; }
    unsigned int getOutputPort() {return outputPort; }

  private:
    unsigned int inputPort, outputPort;
};

/**
 * An TDMA implementation of the MAC Relay Unit that assumes a shared memory and
 * N CPUs in the switch. The CPUs process frames from a single shared queue.
 */
class INET_API MACRelayUnitNPTDMA : public MACRelayUnitNP
{
  public:
    MACRelayUnitNPTDMA();
    ~MACRelayUnitNPTDMA();

  protected:
    /** @name Redefined cSimpleModule member functions. */
    void initialize();
    /**
     * Updates address table with source address, determines output port
     * and sends out (or broadcasts) frame on ports. Includes calls to
     * updateTableWithAddress() and getPortForAddress().
     *
     * The message pointer should not be referenced any more after this call.
     */
    void handleAndDispatchFrame(EtherFrame *frame, const int inputport);
    /**
     * Calls handleIncomingFrame() for frames arrived from outside,
     * and processFrame() for self messages.
     */
    void handleMessage(cMessage *msg);
    /**
     * Get the output port from the schedule description.
    */
    int getPortFromSchedule(const unsigned int inputPort);
    /**
     * Utility function: sends the frame on all ports except inputport.
     * The message pointer should not be referenced any more after this call.
     */
    void broadcastFrame(EtherFrame *frame, const int inputport);
    /**
     * Writes statistics.
     */
    void finish();

private:
    simtime_t calcSlotBoundary() const;
    bool isInputPortRealTime(const unsigned int port) const;
    bool isInputPortRealTime(const unsigned int port, const unsigned int slot) const;
    bool isOutputPortRealTime(const unsigned int port) const;
    bool isOutputPortRealTime(const unsigned int port, const unsigned int slot) const;

    // Parameters controlling how the switch operates in TDMA mode
    unsigned int timeSlotCount;   // how many time slots has a TDMA cycle
    simtime_t timeSlotLength;     // time slot length of the TDMA cycle
    unsigned int actualSlot;      // the actual time slot
    unsigned int missedSlots;     // how many slots are missed
    cMessage * slotTimeEvent;
    bool isSlotted;               // is this deviced slotted
    bool schedulePresent;         // is a schedule present
    unsigned int noRealTimeSlot;
    std::vector<std::vector<IOHolder> > * schedule;
    simtime_t safetyInterval;
    simtime_t interFrameGap;
    cOutVector missingSlots; // record when are slots missed
};

#endif
